<?php
 $page_title = "Laporan Reminder";
 $page_subtitle = "Lihat dan kelola semua reminder yang telah dibuat";

require_once '../includes/header.php';

// 获取筛选参数
 $filter_date_from = isset($_GET['date_from']) ? sanitize_input($_GET['date_from']) : date('Y-m-01'); // 默认为本月第一天
 $filter_date_to = isset($_GET['date_to']) ? sanitize_input($_GET['date_to']) : date('Y-m-d'); // 默认为今天
 $filter_contactable = isset($_GET['contactable']) ? sanitize_input($_GET['contactable']) : 'all';
 $filter_reason = isset($_GET['reason']) ? sanitize_input($_GET['reason']) : 'all';
 $search_query = isset($_GET['search_query']) ? sanitize_input($_GET['search_query']) : '';

// 构建查询条件
 $where_conditions = ["DATE(created_at) BETWEEN ? AND ?"];
 $params = [$filter_date_from, $filter_date_to];
 $types = "ss";

if ($filter_contactable !== 'all') {
    $where_conditions[] = "contactable = ?";
    $params[] = $filter_contactable;
    $types .= "s";
}

if ($filter_reason !== 'all') {
    $where_conditions[] = "reason = ?";
    $params[] = $filter_reason;
    $types .= "s";
}

if (!empty($search_query)) {
    $where_conditions[] = "(police_number LIKE ? OR customer_name LIKE ? OR vin_number LIKE ? OR no_mesin LIKE ?)";
    $searchParam = "%$search_query%";
    $params[] = $searchParam;
    $params[] = $searchParam;
    $params[] = $searchParam;
    $params[] = $searchParam;
    $types .= "ssss";
}

// 构建完整查询
 $sql = "SELECT * FROM reminders WHERE " . implode(' AND ', $where_conditions) . " ORDER BY created_at DESC";
 $stmt = $conn->prepare($sql);

if (!empty($params)) {
    $stmt->bind_param($types, ...$params);
}
 $stmt->execute();
 $result = $stmt->get_result();

// 获取所有唯一的原因选项
 $sql_reasons = "SELECT DISTINCT reason FROM reminders ORDER BY reason";
 $result_reasons = $conn->query($sql_reasons);
 $reason_options = ['all' => 'Semua Alasan'];
if ($result_reasons && $result_reasons->num_rows > 0) {
    while ($row = $result_reasons->fetch_assoc()) {
        $reason_options[$row['reason']] = $row['reason'];
    }
}

// 计算统计数据
 $total_reminders = 0;
 $contactable_yes = 0;
 $contactable_no = 0;
 $booking_count = 0;

if ($result && $result->num_rows > 0) {
    $total_reminders = $result->num_rows;
    
    // 重置指针并计算统计数据
    $result->data_seek(0);
    while ($row = $result->fetch_assoc()) {
        if ($row['contactable'] === 'yes') {
            $contactable_yes++;
        } else {
            $contactable_no++;
        }
        
        if ($row['booking_date_time']) {
            $booking_count++;
        }
    }
    
    // 重置指针以便显示数据
    $result->data_seek(0);
}
?>

<!-- Filter Section -->
<div class="glass-card rounded-xl p-6 mb-6 fade-in">
    <div class="flex flex-col md:flex-row md:items-center md:justify-between mb-4">
        <h2 class="text-lg font-semibold text-gray-800">Filter Laporan</h2>
    </div>
    
    <form id="filterForm" class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
        <div>
            <label class="block text-sm font-medium text-gray-700 mb-1">Tanggal Dari</label>
            <input type="date" id="dateFrom" name="date_from" value="<?php echo $filter_date_from; ?>" 
                   class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-600">
        </div>
        
        <div>
            <label class="block text-sm font-medium text-gray-700 mb-1">Tanggal Sampai</label>
            <input type="date" id="dateTo" name="date_to" value="<?php echo $filter_date_to; ?>" 
                   class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-600">
        </div>
        
        <div>
            <label class="block text-sm font-medium text-gray-700 mb-1">Status Kontak</label>
            <select id="contactable" name="contactable" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-600">
                <option value="all" <?php echo $filter_contactable === 'all' ? 'selected' : ''; ?>>Semua</option>
                <option value="yes" <?php echo $filter_contactable === 'yes' ? 'selected' : ''; ?>>Bisa Dihubungi</option>
                <option value="no" <?php echo $filter_contactable === 'no' ? 'selected' : ''; ?>>Tidak Bisa Dihubungi</option>
            </select>
        </div>
        
        <div>
            <label class="block text-sm font-medium text-gray-700 mb-1">Alasan</label>
            <select id="reason" name="reason" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-600">
                <?php foreach ($reason_options as $value => $label): ?>
                    <option value="<?php echo $value; ?>" <?php echo $filter_reason === $value ? 'selected' : ''; ?>><?php echo $label; ?></option>
                <?php endforeach; ?>
            </select>
        </div>
        
        <div class="lg:col-span-2">
            <label class="block text-sm font-medium text-gray-700 mb-1">Pencarian</label>
            <input type="text" id="searchQuery" name="search_query" value="<?php echo $search_query; ?>" 
                   placeholder="No. Polisi, Nama Customer, No. Rangka, atau No. Mesin" 
                   class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-600">
        </div>
        
        <div class="flex items-end space-x-2 lg:col-span-2">
            <button type="submit" class="flex-1 px-4 py-2 bg-blue-600 hover:bg-blue-700 text-white font-semibold rounded-lg shadow-md transition duration-150 ease-in-out">
                <i class="fas fa-filter mr-2"></i> Terapkan Filter
            </button>
            <button type="button" onclick="exportReport()" class="px-4 py-2 bg-green-600 hover:bg-green-700 text-white font-semibold rounded-lg shadow-md transition duration-150 ease-in-out">
                <i class="fas fa-download mr-2"></i> Export
            </button>
        </div>
    </form>
</div>

<!-- Statistics Cards -->
<div class="grid grid-cols-1 md:grid-cols-4 gap-4 mb-6">
    <div class="glass-card rounded-xl p-6 fade-in">
        <div class="flex items-center">
            <div class="p-3 bg-blue-100 rounded-full mr-4">
                <i class="fas fa-bell text-blue-600 text-xl"></i>
            </div>
            <div>
                <p class="text-sm text-gray-500">Total Reminder</p>
                <p class="text-2xl font-bold text-gray-800"><?php echo $total_reminders; ?></p>
            </div>
        </div>
    </div>
    
    <div class="glass-card rounded-xl p-6 fade-in">
        <div class="flex items-center">
            <div class="p-3 bg-green-100 rounded-full mr-4">
                <i class="fas fa-phone text-green-600 text-xl"></i>
            </div>
            <div>
                <p class="text-sm text-gray-500">Bisa Dihubungi</p>
                <p class="text-2xl font-bold text-gray-800"><?php echo $contactable_yes; ?></p>
                <p class="text-xs text-gray-500"><?php echo $total_reminders > 0 ? round(($contactable_yes / $total_reminders) * 100, 1) : 0; ?>%</p>
            </div>
        </div>
    </div>
    
    <div class="glass-card rounded-xl p-6 fade-in">
        <div class="flex items-center">
            <div class="p-3 bg-red-100 rounded-full mr-4">
                <i class="fas fa-phone-slash text-red-600 text-xl"></i>
            </div>
            <div>
                <p class="text-sm text-gray-500">Tidak Bisa Dihubungi</p>
                <p class="text-2xl font-bold text-gray-800"><?php echo $contactable_no; ?></p>
                <p class="text-xs text-gray-500"><?php echo $total_reminders > 0 ? round(($contactable_no / $total_reminders) * 100, 1) : 0; ?>%</p>
            </div>
        </div>
    </div>
    
    <div class="glass-card rounded-xl p-6 fade-in">
        <div class="flex items-center">
            <div class="p-3 bg-purple-100 rounded-full mr-4">
                <i class="fas fa-calendar-check text-purple-600 text-xl"></i>
            </div>
            <div>
                <p class="text-sm text-gray-500">Booking</p>
                <p class="text-2xl font-bold text-gray-800"><?php echo $booking_count; ?></p>
                <p class="text-xs text-gray-500"><?php echo $total_reminders > 0 ? round(($booking_count / $total_reminders) * 100, 1) : 0; ?>%</p>
            </div>
        </div>
    </div>
</div>

<!-- Reminder Table -->
<div class="glass-card rounded-xl p-6 mb-6 fade-in">
    <div class="flex items-center justify-between mb-4">
        <h2 class="text-lg font-semibold text-gray-800">Data Reminder</h2>
        <div class="text-sm text-gray-500">
            Menampilkan <?php echo $result ? $result->num_rows : 0; ?> reminder
        </div>
    </div>

    <div class="overflow-x-auto">
        <?php if ($result && $result->num_rows > 0): ?>
            <table class="w-full">
                <thead>
                    <tr class="text-left text-xs text-gray-500 uppercase tracking-wider border-b">
                        <th class="pb-3">No</th>
                        <th class="pb-3">Tanggal Dibuat</th>
                        <th class="pb-3">No. Polisi</th>
                        <th class="pb-3">Nama Customer</th>
                        <th class="pb-3">No. Telepon</th>
                        <th class="pb-3">Kontak</th>
                        <th class="pb-3">Alasan</th>
                        <th class="pb-3">Booking</th>
                        <th class="pb-3">Dibuat Oleh</th>
                        <th class="pb-3">Aksi</th>
                    </tr>
                </thead>
                <tbody class="text-sm">
                    <?php 
                    $no = 1;
                    while ($row = $result->fetch_assoc()): 
                    ?>
                        <tr class="table-row border-b hover:bg-gray-50">
                            <td class="py-3"><?php echo $no++; ?></td>
                            <td class="py-3">
                                <p class="text-gray-800"><?php echo date('d M Y H:i', strtotime($row['created_at'])); ?></p>
                                <p class="text-xs text-gray-500"><?php echo time_elapsed_string($row['created_at']); ?></p>
                            </td>
                            <td class="py-3">
                                <p class="font-medium text-gray-800"><?php echo $row['police_number']; ?></p>
                            </td>
                            <td class="py-3">
                                <p class="font-medium text-gray-800"><?php echo $row['customer_name']; ?></p>
                            </td>
                            <td class="py-3">
                                <p class="text-gray-800"><?php echo $row['phone_number'] ?? '-'; ?></p>
                            </td>
                            <td class="py-3">
                                <?php if ($row['contactable'] === 'yes'): ?>
                                    <span class="px-2 py-1 text-xs rounded-full bg-green-100 text-green-800">Bisa Dihubungi</span>
                                <?php else: ?>
                                    <span class="px-2 py-1 text-xs rounded-full bg-red-100 text-red-800">Tidak Bisa Dihubungi</span>
                                <?php endif; ?>
                            </td>
                            <td class="py-3">
                                <p class="text-gray-800"><?php echo $row['reason']; ?></p>
                            </td>
                            <td class="py-3">
                                <?php if ($row['booking_date_time']): ?>
                                    <p class="text-gray-800"><?php echo date('d M Y H:i', strtotime($row['booking_date_time'])); ?></p>
                                <?php else: ?>
                                    <p class="text-gray-500">-</p>
                                <?php endif; ?>
                            </td>
                            <td class="py-3">
                                <p class="text-gray-800"><?php echo $row['reminder_by'] ?? '-'; ?></p>
                            </td>
                            <td class="py-3">
                                <div class="flex space-x-2">
                                    <button class="text-blue-600 hover:text-blue-800" onclick="viewReminderDetail('<?php echo $row['id']; ?>')" title="Lihat Detail">
                                        <i class="fas fa-eye"></i>
                                    </button>
                                    <button class="text-red-600 hover:text-red-800" onclick="deleteReminder('<?php echo $row['id']; ?>')" title="Hapus">
                                        <i class="fas fa-trash"></i>
                                    </button>
                                </div>
                            </td>
                        </tr>
                    <?php endwhile; ?>
                </tbody>
            </table>
        <?php else: ?>
            <div class="text-center py-8 text-gray-500">
                <i class="fas fa-bell text-4xl mb-2"></i>
                <p class="text-lg">Tidak ada data reminder yang ditemukan</p>
                <p class="text-sm">Coba ubah filter atau gunakan kata kunci pencarian yang lain.</p>
            </div>
        <?php endif; ?>
    </div>
</div>

<!-- Modal untuk Detail Reminder -->
<div id="reminderDetailModal" class="fixed inset-0 bg-black bg-opacity-50 hidden z-50 flex items-center justify-center p-4">
    <div class="bg-white rounded-xl shadow-2xl w-full max-w-4xl max-h-[90vh] flex flex-col">
        <div class="flex justify-between items-center p-6 border-b bg-gradient-to-r from-blue-50 to-indigo-50 rounded-t-xl">
            <h3 class="text-xl font-bold text-gray-800">Detail Reminder</h3>
            <button onclick="closeReminderDetailModal()" class="text-gray-500 hover:text-gray-700 bg-white rounded-full p-2 shadow-md transition-all hover:shadow-lg">
                <i class="fas fa-times text-xl"></i>
            </button>
        </div>
        
        <div id="reminderDetailContent" class="flex-1 overflow-y-auto p-6">
            <div class="flex justify-center items-center py-12">
                <div class="text-center">
                    <i class="fas fa-spinner fa-spin text-4xl text-blue-600 mb-4"></i>
                    <p class="text-gray-600">Memuat data...</p>
                </div>
            </div>
        </div>
        
        <div class="border-t p-4 bg-gray-50 rounded-b-xl">
            <div class="flex justify-end">
                <button onclick="closeReminderDetailModal()" class="px-6 py-2 bg-blue-600 hover:bg-blue-700 text-white font-medium rounded-lg shadow-md transition-all hover:shadow-lg">
                    Tutup
                </button>
            </div>
        </div>
    </div>
</div>

<!-- JavaScript -->
<script>
document.addEventListener('DOMContentLoaded', function() {
    const filterForm = document.getElementById('filterForm');
    if (filterForm) {
        filterForm.addEventListener('submit', function(e) {
            e.preventDefault();
            
            const formData = new FormData(this);
            const params = new URLSearchParams();
            
            for (const [key, value] of formData.entries()) {
                if (value) {
                    params.append(key, value);
                }
            }
            
            window.location.href = '/reminder/report?' + params.toString();
        });
    }
});

function viewReminderDetail(reminderId) {
    document.getElementById('reminderDetailContent').innerHTML = '<div class="flex justify-center items-center py-12"><div class="text-center"><i class="fas fa-spinner fa-spin text-4xl text-blue-600 mb-4"></i><p class="text-gray-600">Memuat data...</p></div></div>';
    document.getElementById('reminderDetailModal').classList.remove('hidden');
    
    fetch('/reminder/get_detail/' + reminderId)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                const reminder = data.reminder;
                
                let html = `
                    <div class="space-y-6">
                        <div class="bg-gray-50 p-4 rounded-lg">
                            <h4 class="font-semibold text-gray-700 mb-3 border-b pb-2">Informasi Reminder</h4>
                            <div class="grid grid-cols-1 md:grid-cols-2 gap-4 text-sm">
                                <div>
                                    <p class="text-xs text-gray-500">ID Reminder</p>
                                    <p class="font-semibold text-gray-800">${reminder.id}</p>
                                </div>
                                <div>
                                    <p class="text-xs text-gray-500">Tanggal Dibuat</p>
                                    <p class="font-semibold text-gray-800">${new Date(reminder.created_at).toLocaleString('id-ID')}</p>
                                </div>
                                <div>
                                    <p class="text-xs text-gray-500">Dibuat Oleh</p>
                                    <p class="font-semibold text-gray-800">${reminder.reminder_by || '-'}</p>
                                </div>
                                <div>
                                    <p class="text-xs text-gray-500">Status Kontak</p>
                                    <p class="font-semibold text-gray-800">
                                        ${reminder.contactable === 'yes' ? 
                                            '<span class="px-2 py-1 text-xs rounded-full bg-green-100 text-green-800">Bisa Dihubungi</span>' : 
                                            '<span class="px-2 py-1 text-xs rounded-full bg-red-100 text-red-800">Tidak Bisa Dihubungi</span>'}
                                    </p>
                                </div>
                                <div>
                                    <p class="text-xs text-gray-500">Alasan</p>
                                    <p class="font-semibold text-gray-800">${reminder.reason}</p>
                                </div>
                                <div>
                                    <p class="text-xs text-gray-500">Booking</p>
                                    <p class="font-semibold text-gray-800">
                                        ${reminder.booking_date_time ? 
                                            new Date(reminder.booking_date_time).toLocaleString('id-ID') : 
                                            '-'}
                                    </p>
                                </div>
                            </div>
                        </div>
                        
                        <div class="bg-gray-50 p-4 rounded-lg">
                            <h4 class="font-semibold text-gray-700 mb-3 border-b pb-2">Data Kendaraan</h4>
                            <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4 text-sm">
                                <div>
                                    <p class="text-xs text-gray-500">No. Polisi</p>
                                    <p class="font-semibold text-gray-800">${reminder.police_number}</p>
                                </div>
                                <div>
                                    <p class="text-xs text-gray-500">Nama Customer</p>
                                    <p class="font-semibold text-gray-800">${reminder.customer_name}</p>
                                </div>
                                <div>
                                    <p class="text-xs text-gray-500">No. Telepon</p>
                                    <p class="font-semibold text-gray-800">${reminder.phone_number || '-'}</p>
                                </div>
                                <div>
                                    <p class="text-xs text-gray-500">No. Rangka (VIN)</p>
                                    <p class="font-semibold text-gray-800">${reminder.vin_number || '-'}</p>
                                </div>
                                <div>
                                    <p class="text-xs text-gray-500">No. Mesin</p>
                                    <p class="font-semibold text-gray-800">${reminder.no_mesin || '-'}</p>
                                </div>
                                <div>
                                    <p class="text-xs text-gray-500">Model Kendaraan</p>
                                    <p class="font-semibold text-gray-800">${reminder.basic_model || '-'}</p>
                                </div>
                                <div>
                                    <p class="text-xs text-gray-500">Type</p>
                                    <p class="font-semibold text-gray-800">${reminder.type || '-'}</p>
                                </div>
                                <div>
                                    <p class="text-xs text-gray-500">Transmisi</p>
                                    <p class="font-semibold text-gray-800">${reminder.transmisi || '-'}</p>
                                </div>
                                <div>
                                    <p class="text-xs text-gray-500">Kategori</p>
                                    <p class="font-semibold text-gray-800">${reminder.kategori || '-'}</p>
                                </div>
                                <div>
                                    <p class="text-xs text-gray-500">Tahun</p>
                                    <p class="font-semibold text-gray-800">${reminder.tahun || '-'}</p>
                                </div>
                                <div>
                                    <p class="text-xs text-gray-500">Warna</p>
                                    <p class="font-semibold text-gray-800">${reminder.warna || '-'}</p>
                                </div>
                                <div>
                                    <p class="text-xs text-gray-500">No. Buku Service</p>
                                    <p class="font-semibold text-gray-800">${reminder.no_buku_service || '-'}</p>
                                </div>
                                <div>
                                    <p class="text-xs text-gray-500">Tahun Registrasi</p>
                                    <p class="font-semibold text-gray-800">${reminder.tahun_registrasi || '-'}</p>
                                </div>
                                <div class="md:col-span-2 lg:col-span-1">
                                    <p class="text-xs text-gray-500">Alamat</p>
                                    <p class="font-semibold text-gray-800">${reminder.alamat || '-'}</p>
                                </div>
                            </div>
                        </div>
                    </div>
                `;
                
                document.getElementById('reminderDetailContent').innerHTML = html;
            } else {
                document.getElementById('reminderDetailContent').innerHTML = `
                    <div class="flex flex-col items-center justify-center py-12 text-red-500">
                        <i class="fas fa-exclamation-circle text-4xl mb-4"></i>
                        <p class="text-lg font-medium">Gagal memuat data reminder</p>
                        <p class="text-sm">${data.message || 'Terjadi kesalahan saat memuat data'}</p>
                    </div>
                `;
            }
        })
        .catch(error => {
            console.error('Error:', error);
            document.getElementById('reminderDetailContent').innerHTML = `
                <div class="flex flex-col items-center justify-center py-12 text-red-500">
                    <i class="fas fa-exclamation-circle text-4xl mb-4"></i>
                    <p class="text-lg font-medium">Terjadi kesalahan saat memuat data</p>
                    <p class="text-sm">Periksa console browser (F12) untuk detail error</p>
                </div>
            `;
        });
}

function closeReminderDetailModal() {
    document.getElementById('reminderDetailModal').classList.add('hidden');
}

function deleteReminder(reminderId) {
    if (confirm('Apakah Anda yakin ingin menghapus reminder ini?')) {
        fetch('/reminder/delete/' + reminderId, {
            method: 'DELETE'
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showToast('Reminder berhasil dihapus', 'success');
                setTimeout(() => {
                    window.location.reload();
                }, 1000);
            } else {
                showToast('Gagal menghapus reminder: ' + (data.message || 'Terjadi kesalahan'), 'error');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            showToast('Terjadi kesalahan saat menghapus reminder', 'error');
        });
    }
}

function exportReport() {
    const dateFrom = document.getElementById('dateFrom').value;
    const dateTo = document.getElementById('dateTo').value;
    const contactable = document.getElementById('contactable').value;
    const reason = document.getElementById('reason').value;
    const searchQuery = document.getElementById('searchQuery').value;
    
    const params = new URLSearchParams();
    if (dateFrom) params.append('date_from', dateFrom);
    if (dateTo) params.append('date_to', dateTo);
    if (contactable !== 'all') params.append('contactable', contactable);
    if (reason !== 'all') params.append('reason', reason);
    if (searchQuery) params.append('search_query', searchQuery);
    
    window.location.href = '/reminder/export?' + params.toString();
}
</script>

<?php require_once '../includes/footer.php'; ?>