<?php
 $page_title = "Pencarian Data Penjualan";
 $page_subtitle = "Cari data penjualan berdasarkan nomor rangka, mesin, polisi, atau nama pemilik";

require_once '../includes/header.php';

function time_elapsed_string($datetime) {
    $now = new DateTime;
    $ago = new DateTime($datetime);
    $diff = $now->diff($ago);

    if ($diff->y > 0) {
        return $diff->y . ' tahun yang lalu';
    }
    if ($diff->m > 0) {
        return $diff->m . ' bulan yang lalu';
    }
    if ($diff->d > 0) {
        return $diff->d . ' hari yang lalu';
    }
    
    return "Hari ini";
}

 $reasons_from_db = ['yes' => [], 'no' => []];
 $sql_reasons = "SELECT reason_text, category FROM reminder_reasons WHERE is_active = 1 ORDER BY category, reason_text";
 $result_reasons = $conn->query($sql_reasons);

if ($result_reasons && $result_reasons->num_rows > 0) {
    while ($row = $result_reasons->fetch_assoc()) {
        $reasons_from_db[$row['category']][] = $row['reason_text'];
    }
} else {
    $reasons_from_db = [
        'yes' => [
            "Booking", "Akan Datang Langsung", "Akan Konfirmasi Kembali", "Baru saja Service di Bengkel Suzuki",
            "Belum Ada Dana", "Belum Ada Waktu Service", "Belum Check KM/Odometer", "Bukan Mobil Suzuki",
            "Bukan Nomor Telepon Pemilik Mobil, Hanya Bawa Service Saja", "Buku Service Hilang",
            "Dalam Perbaikan / Sedang Service saat Ini", "Harga Servis Mahal", "KM Belum Tercapai", "Kupon Habis",
            "Lokasi Bengkel Jauh", "Masalah Pelayanan", "Masih diluar kota / dalam perjalanan", "Nomor Tidak Aktif",
            "Menunggu Plate Nomor", "Menunggu Sparepart", "Mobil Ditarik Leasing", "Mobil Hilang",
            "Mobil Masih Digunakan / Disewa", "Mobil Sudah Dijual", "Nomor Telepon Supir, dan Supir Sudah Tidak Kerja Lagi",
            "Pindah", "Servis di Bengkel Umum", "Tidak Ada Yang Bisa Bawa Kendaraan", "Tidak Akan Service Lagi",
            "Tunggu Promo / Minta Harga", "Waktu Servis Lama", "SMS/ WhatsApp", "Lain-Lain"
        ],
        'no' => [
            "Diluar jangkauan/Tidak dapat dihubungi", "Mailbox", "No. Telp Salah", "No. Telp Sales-force",
            "Nomor diblokir", "Nomor HP tidak ada", "Nomor tidak terdaftar", "Panggilan dialihkan",
            "Salah Sambung", "Telp. Sibuk", "Telp. Tidak Aktif", "Telp. Tidak Diangkat",
            "Konsumen Tidak Ditempat", "Meeting / Tidak Bisa Diganggu", "Pelanggan Sibuk",
            "Sedang Dirawat / Di Rumah Sakit", "Telepon Ditolak", "Telepon Terputus", "Jangan Telepon Lagi"
        ]
    ];
}

 $all_details_for_modal = [];

 $search = '';
if (isset($_GET['search_query'])) {
    $decoded_search = base64_decode($_GET['search_query']);
    if ($decoded_search !== false) {
        $search = sanitize_input($decoded_search);
    }
}

 $searchPerformed = !empty($search);
 $result = null;
 $lastServiceResult = null;
 $reminderResult = null;

if ($searchPerformed) {
    $sql_last = "SELECT * FROM penjualan 
                 WHERE `No. Polisi` LIKE ? OR `No. Rangka` LIKE ? OR `No. Mesin` LIKE ? 
                 ORDER BY `Tanggal Finish SPK` DESC 
                 LIMIT 1";
    
    $stmt_last = $conn->prepare($sql_last);
    $searchParam = "%$search%";
    $stmt_last->bind_param("sss", $searchParam, $searchParam, $searchParam);
    $stmt_last->execute();
    $lastServiceResult = $stmt_last->get_result();

    $sql = "SELECT 
            `No. SPK`,
            MAX(`Tanggal Finish SPK`) as `Tanggal Selesai`,
            `Nama Customer`,
            `No. Polisi`,
            `Model Kendaraan`,
            COUNT(*) as `Jumlah Item`,
            SUM(`Total + PPN`) as `Total Biaya`
            FROM penjualan 
            WHERE `Nama Pemilik STNK` LIKE ? OR `No. Polisi` LIKE ? OR `No. Rangka` LIKE ? OR `No. Mesin` LIKE ? 
            GROUP BY `No. SPK`
            ORDER BY `Tanggal Selesai` DESC";

    $stmt = $conn->prepare($sql);
    $stmt->bind_param("ssss", $searchParam, $searchParam, $searchParam, $searchParam);
    $stmt->execute();
    $result = $stmt->get_result();

    // Query untuk reminder history
    $sql_reminder = "SELECT * FROM reminders 
                     WHERE police_number LIKE ? OR vin_number LIKE ? OR no_mesin LIKE ? 
                     ORDER BY created_at DESC";
    
    $stmt_reminder = $conn->prepare($sql_reminder);
    $stmt_reminder->bind_param("sss", $searchParam, $searchParam, $searchParam);
    $stmt_reminder->execute();
    $reminderResult = $stmt_reminder->get_result();
    
    // Get the latest reminder for this vehicle to check if it's older than 40 days
    $latestReminder = null;
    if ($reminderResult && $reminderResult->num_rows > 0) {
        $reminderResult->data_seek(0);
        $latestReminder = $reminderResult->fetch_assoc();
        $reminderResult->data_seek(0); // Reset pointer for later use
    }

    if ($result && $result->num_rows > 0) {
        $spk_list = [];
        while ($row = $result->fetch_assoc()) {
            $spk_list[] = $row['No. SPK'];
        }

        if (!empty($spk_list)) {
            $spk_placeholders = implode(',', array_fill(0, count($spk_list), '?'));
            $types = str_repeat('s', count($spk_list));

            $sql_details = "SELECT * FROM penjualan WHERE `No. SPK` IN ($spk_placeholders)";
            $stmt_details = $conn->prepare($sql_details);
            $stmt_details->bind_param($types, ...$spk_list);
            $stmt_details->execute();
            $details_result = $stmt_details->get_result();

            while ($detail_row = $details_result->fetch_assoc()) {
                $all_details_for_modal[$detail_row['No. SPK']][] = $detail_row;
            }
        }
        $result->data_seek(0);
    }
}

// Function to check if reminder button should be enabled
function canCreateReminder($latestReminder) {
    if (!$latestReminder) {
        // No reminder exists, can create one
        return true;
    }
    
    $now = new DateTime();
    $reminderDate = new DateTime($latestReminder['created_at']);
    $interval = $now->diff($reminderDate);
    
    // Check if more than 40 days have passed
    return $interval->days > 40;
}

 $canCreateReminder = canCreateReminder($latestReminder);
?>
<!-- Filter Section -->
<div class="glass-card rounded-xl p-6 mb-6 fade-in">
    <div class="flex flex-col md:flex-row md:items-center md:justify-between mb-4">
        <h2 class="text-lg font-semibold text-gray-800">Pencarian Data Penjualan</h2>
    </div>
    
    <form id="searchForm" class="grid grid-cols-1 md:grid-cols-2 gap-4">
        <div>
            <label class="block text-sm font-medium text-gray-700 mb-1">Kata Kunci Pencarian</label>
            <input type="text" id="searchInput" name="search" value="<?php echo htmlspecialchars($search); ?>" 
                   placeholder="Nama Pemilik, No. Polisi, No. Rangka, atau No. Mesin" 
                   class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-600">
        </div>
        
        <div class="flex items-end">
            <button type="submit" class="w-full px-4 py-2 bg-blue-600 hover:bg-blue-700 text-white font-semibold rounded-lg shadow-md transition duration-150 ease-in-out">
                <i class="fas fa-search mr-2"></i> Cari Data
            </button>
        </div>
    </form>
</div>

<?php if ($searchPerformed): ?>
    <!-- Service Terakhir Section -->
    <div class="glass-card rounded-xl p-6 mb-6 fade-in">
        <div class="flex items-center mb-4">
            <h2 class="text-lg font-semibold text-gray-800">Data Kendaraan (Service Terakhir)</h2>
        </div>

        <?php if ($lastServiceResult && $lastServiceResult->num_rows > 0): ?>
            <?php $row = $lastServiceResult->fetch_assoc(); ?>
            <div class="p-4 bg-green-50 border border-green-200 rounded-lg">
                <h3 class="font-semibold text-green-800 mb-3">Data Ditemukan</h3>
                <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4 text-sm">
                    <div>
                        <p class="text-gray-600">Nama Customer</p>
                        <p class="font-medium"><?php echo $row['Nama Customer'] ?? '-'; ?></p>
                    </div>
                    <div>
                        <p class="text-gray-600">No. Telepon</p>
                        <p class="font-medium"><?php echo $row['No. Telepon'] ?? '-'; ?></p>
                    </div>
                    <div>
                        <p class="text-gray-600">Nama Pemilik STNK</p>
                        <p class="font-medium"><?php echo $row['Nama Pemilik STNK'] ?? '-'; ?></p>
                    </div>
                    <div class="md:col-span-2 lg:col-span-1">
                        <p class="text-gray-600">Alamat Pemilik STNK</p>
                        <p class="font-medium"><?php echo $row['Alamat Pemilik'] ?? '-'; ?></p>
                    </div>
                    <div>
                        <p class="text-gray-600">KM Terakhir</p>
                        <p class="font-medium"><?php echo number_format($row['KM'], 0, ',', '.'); ?></p>
                    </div>
                    <div>
                        <p class="text-gray-600">No. Polisi</p>
                        <p class="font-medium"><?php echo $row['No. Polisi']; ?></p>
                    </div>
                    <div>
                        <p class="text-gray-600">Type</p>
                        <p class="font-medium"><?php echo $row['Type'] ?? '-'; ?></p>
                    </div>
                    <div>
                        <p class="text-gray-600">Model Kendaraan</p>
                        <p class="font-medium"><?php echo $row['Model Kendaraan']; ?></p>
                    </div>
                    <div>
                        <p class="text-gray-600">Transmisi</p>
                        <p class="font-medium"><?php echo $row['Transmisi'] ?? '-'; ?></p>
                    </div>
                    <div>
                        <p class="text-gray-600">Kategori</p>
                        <p class="font-medium"><?php echo $row['Kategori'] ?? '-'; ?></p>
                    </div>
                    <div>
                        <p class="text-gray-600">Tahun</p>
                        <p class="font-medium"><?php echo $row['Tahun'] ?? '-'; ?></p>
                    </div>
                    <div>
                        <p class="text-gray-600">No. Buku Service</p>
                        <p class="font-medium"><?php echo $row['No. Buku Service'] ?? '-'; ?></p>
                    </div>
                    <div>
                        <p class="text-gray-600">Warna</p>
                        <p class="font-medium"><?php echo $row['Warna'] ?? '-'; ?></p>
                    </div>
                    <div>
                        <p class="text-gray-600">Tahun Registrasi</p>
                        <p class="font-medium"><?php echo $row['Tahun Registrasi'] ?? '-'; ?></p>
                    </div>
                    <div>
                        <p class="text-gray-600">No. Rangka (VIN)</p>
                        <p class="font-medium"><?php echo $row['No. Rangka']; ?></p>
                    </div>
                    <div>
                        <p class="text-gray-600">No. Mesin</p>
                        <p class="font-medium"><?php echo $row['No. Mesin']; ?></p>
                    </div>
                    <div>
                        <p class="text-gray-600">SPK Terakhir</p>
                        <div class="flex items-center">
                            <p class="font-medium"><?php echo $row['No. SPK'] ?? '-'; ?></p>
                            <button class="ml-2 text-blue-600 hover:text-blue-800" onclick="viewPenjualanBySpk('<?php echo $row['No. SPK']; ?>')" title="Lihat Detail">
                                <i class="fas fa-eye"></i>
                            </button>
                        </div>
                    </div>
                    <div>
                        <p class="text-gray-600">Tgl. Serv. Terakhir</p>
                        <p class="font-medium"><?php echo date('d M Y', strtotime($row['Tanggal Finish SPK'])); ?></p>
                    </div>
                </div>
                
                <?php if (!empty($row['Rekomendasi'])): ?>
                <div class="mt-4 pt-4 border-t border-green-200">
                    <p class="text-sm text-gray-600 font-medium">Rekomendasi:</p>
                    <p class="text-sm text-gray-800 mt-1"><?php echo htmlspecialchars($row['Rekomendasi']); ?></p>
                </div>
                <?php endif; ?>

                <div class="mt-4">
                    <?php if ($canCreateReminder): ?>
                    <button class="w-full px-4 py-2 bg-green-600 hover:bg-green-700 text-white font-semibold rounded-lg shadow-md transition duration-150 ease-in-out transform hover:scale-105" onclick="openReminderModal(
                        '<?php echo $row['No. SPK']; ?>', 
                        '<?php echo htmlspecialchars($row['Nama Customer'] ?? ''); ?>', 
                        '<?php echo $row['No. Rangka'] ?? ''; ?>',
                        '<?php echo $row['No. Polisi']; ?>',
                        '<?php echo $row['No. Telepon'] ?? ''; ?>',
                        '<?php echo htmlspecialchars($row['Alamat Pemilik'] ?? ''); ?>',
                        '<?php echo $row['Model Kendaraan']; ?>',
                        '<?php echo $row['Type'] ?? ''; ?>',
                        '<?php echo $row['Transmisi'] ?? ''; ?>',
                        '<?php echo $row['Kategori'] ?? ''; ?>',
                        '<?php echo $row['Tahun'] ?? ''; ?>',
                        '<?php echo $row['Warna'] ?? ''; ?>',
                        '<?php echo $row['No. Mesin'] ?? ''; ?>',
                        '<?php echo $row['No. Buku Service'] ?? ''; ?>',
                        '<?php echo $row['Tahun Registrasi'] ?? ''; ?>'
                    )">
                        <i class="fas fa-bell mr-2"></i> BUAT REMINDER
                    </button>
                    <?php else: ?>
                    <div class="w-full px-4 py-2 bg-gray-400 text-white font-semibold rounded-lg shadow-md cursor-not-allowed" title="Reminder hanya dapat dibuat setelah 40 hari dari reminder terakhir">
                        <i class="fas fa-bell-slash mr-2"></i> REMINDER TIDAK TERSEDIA
                        <?php if ($latestReminder): ?>
                        <p class="text-xs mt-1">Reminder terakhir: <?php echo date('d M Y', strtotime($latestReminder['created_at'])); ?> (<?php echo time_elapsed_string($latestReminder['created_at']); ?>)</p>
                        <?php endif; ?>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
        <?php else: ?>
            <div class="p-4 bg-red-50 border border-red-200 rounded-lg">
                <p class="text-red-700">Tidak ditemukan riwayat service untuk kendaraan tersebut.</p>
            </div>
        <?php endif; ?>
    </div>

    <!-- Penjualan Table (Rekap per SPK) -->
    <div class="glass-card rounded-xl p-6 mb-6 fade-in">
        <div class="flex items-center justify-between mb-4">
            <h2 class="text-lg font-semibold text-gray-800">History Service</h2>
            <?php if ($result): ?>
                <div class="text-sm text-gray-500">
                    Menampilkan <?php echo $result->num_rows; ?> SPK untuk "<strong><?php echo htmlspecialchars($search); ?></strong>"
                </div>
            <?php endif; ?>
        </div>

        <div class="overflow-x-auto">
            <?php if ($result && $result->num_rows > 0): ?>
                <table class="w-full">
                    <thead>
                        <tr class="text-left text-xs text-gray-500 uppercase tracking-wider border-b">
                            <th class="pb-3">No</th>
                            <th class="pb-3">Tanggal Selesai</th>
                            <th class="pb-3">No. SPK</th>
                            <th class="pb-3">Nama Customer</th>
                            <th class="pb-3">No. Polisi</th>
                            <th class="pb-3">Model Kendaraan</th>
                            <th class="pb-3">Jumlah Item</th>
                            <th class="pb-3">Total Biaya</th>
                            <th class="pb-3">Aksi</th>
                        </tr>
                    </thead>
                    <tbody class="text-sm">
                        <?php 
                        $no = 1;
                        while ($row = $result->fetch_assoc()): 
                        ?>
                            <tr class="table-row border-b hover:bg-gray-50">
                                <td class="py-3"><?php echo $no++; ?></td>
                                <td class="py-3">
                                    <p class="text-gray-800"><?php echo date('d M Y', strtotime($row['Tanggal Selesai'])); ?></p>
                                </td>
                                <td class="py-3">
                                    <p class="font-medium text-gray-800"><?php echo $row['No. SPK']; ?></p>
                                </td>
                                <td class="py-3">
                                    <p class="font-medium text-gray-800"><?php echo $row['Nama Customer'] ?? '-'; ?></p>
                                </td>
                                <td class="py-3">
                                    <p class="text-gray-800"><?php echo $row['No. Polisi']; ?></p>
                                </td>
                                <td class="py-3">
                                    <p class="text-gray-800"><?php echo $row['Model Kendaraan']; ?></p>
                                </td>
                                <td class="py-3 text-center">
                                    <span class="px-2 py-1 text-xs rounded-full bg-gray-100 text-gray-700"><?php echo $row['Jumlah Item']; ?></span>
                                </td>
                                <td class="py-3">
                                    <p class="font-semibold text-gray-800">Rp <?php echo number_format($row['Total Biaya'], 0, ',', '.'); ?></p>
                                </td>
                                <td class="py-3">
                                    <div class="flex space-x-2">
                                        <?php if ($canCreateReminder): ?>
                                        <button class="text-green-600 hover:text-green-800" onclick="openReminderModal(
                                            '<?php echo $row['No. SPK']; ?>', 
                                            '<?php echo htmlspecialchars($row['Nama Customer'] ?? ''); ?>', 
                                            '<?php echo $row['No. Rangka'] ?? ''; ?>',
                                            '<?php echo $row['No. Polisi']; ?>',
                                            '<?php echo $row['No. Telepon'] ?? ''; ?>',
                                            '<?php echo htmlspecialchars($row['Alamat Pemilik'] ?? ''); ?>',
                                            '<?php echo $row['Model Kendaraan']; ?>',
                                            '<?php echo $row['Type'] ?? ''; ?>',
                                            '<?php echo $row['Transmisi'] ?? ''; ?>',
                                            '<?php echo $row['Kategori'] ?? ''; ?>',
                                            '<?php echo $row['Tahun'] ?? ''; ?>',
                                            '<?php echo $row['Warna'] ?? ''; ?>',
                                            '<?php echo $row['No. Mesin'] ?? ''; ?>',
                                            '<?php echo $row['No. Buku Service'] ?? ''; ?>',
                                            '<?php echo $row['Tahun Registrasi'] ?? ''; ?>'
                                        )" title="Buat Reminder">
                                            <i class="fas fa-bell"></i>
                                        </button>
                                        <?php else: ?>
                                        <button class="text-gray-400 cursor-not-allowed" title="Reminder hanya dapat dibuat setelah 40 hari dari reminder terakhir">
                                            <i class="fas fa-bell-slash"></i>
                                        </button>
                                        <?php endif; ?>
                                        <button class="text-blue-600 hover:text-blue-800" onclick="viewPenjualanBySpk('<?php echo $row['No. SPK']; ?>')" title="Lihat Detail">
                                            <i class="fas fa-eye"></i>
                                        </button>
                                    </div>
                                </td>
                            </tr>
                        <?php endwhile; ?>
                    </tbody>
                </table>
            <?php else: ?>
                <div class="text-center py-8 text-gray-500">
                    <i class="fas fa-search text-4xl mb-2"></i>
                    <p class="text-lg">Tidak ada data rekap SPK yang ditemukan</p>
                    <p class="text-sm">Coba gunakan kata kunci pencarian yang lain.</p>
                </div>
            <?php endif; ?>
        </div>
    </div>

    <!-- Reminder History Table -->
    <div class="glass-card rounded-xl p-6 mb-6 fade-in">
        <div class="flex items-center justify-between mb-4">
            <h2 class="text-lg font-semibold text-gray-800">History Reminder</h2>
            <?php if ($reminderResult): ?>
                <div class="text-sm text-gray-500">
                    Menampilkan <?php echo $reminderResult->num_rows; ?> reminder untuk "<strong><?php echo htmlspecialchars($search); ?></strong>"
                </div>
            <?php endif; ?>
        </div>

        <div class="overflow-x-auto">
            <?php if ($reminderResult && $reminderResult->num_rows > 0): ?>
                <table class="w-full">
                    <thead>
                        <tr class="text-left text-xs text-gray-500 uppercase tracking-wider border-b">
                            <th class="pb-3">No</th>
                            <th class="pb-3">Tanggal Dibuat</th>
                            <th class="pb-3">No. Polisi</th>
                            <th class="pb-3">Nama Customer</th>
                            <th class="pb-3">Kontak</th>
                            <th class="pb-3">Alasan</th>
                            <th class="pb-3">Booking</th>
                            <th class="pb-3">Dibuat Oleh</th>
                        </tr>
                    </thead>
                    <tbody class="text-sm">
                        <?php 
                        $no = 1;
                        while ($row = $reminderResult->fetch_assoc()): 
                        ?>
                            <tr class="table-row border-b hover:bg-gray-50">
                                <td class="py-3"><?php echo $no++; ?></td>
                                <td class="py-3">
                                    <p class="text-gray-800"><?php echo date('d M Y H:i', strtotime($row['created_at'])); ?></p>
                                    <p class="text-xs text-gray-500"><?php echo time_elapsed_string($row['created_at']); ?></p>
                                </td>
                                <td class="py-3">
                                    <p class="font-medium text-gray-800"><?php echo $row['police_number']; ?></p>
                                </td>
                                <td class="py-3">
                                    <p class="font-medium text-gray-800"><?php echo $row['customer_name']; ?></p>
                                </td>
                                <td class="py-3">
                                    <?php if ($row['contactable'] === 'yes'): ?>
                                        <span class="px-2 py-1 text-xs rounded-full bg-green-100 text-green-800">Bisa Dihubungi</span>
                                    <?php else: ?>
                                        <span class="px-2 py-1 text-xs rounded-full bg-red-100 text-red-800">Tidak Bisa Dihubungi</span>
                                    <?php endif; ?>
                                </td>
                                <td class="py-3">
                                    <p class="text-gray-800"><?php echo $row['reason']; ?></p>
                                </td>
                                <td class="py-3">
                                    <?php if ($row['booking_date_time']): ?>
                                        <p class="text-gray-800"><?php echo date('d M Y H:i', strtotime($row['booking_date_time'])); ?></p>
                                    <?php else: ?>
                                        <p class="text-gray-500">-</p>
                                    <?php endif; ?>
                                </td>
                                <td class="py-3">
                                    <p class="text-gray-800"><?php echo $row['reminder_by'] ?? '-'; ?></p>
                                </td>
                            </tr>
                        <?php endwhile; ?>
                    </tbody>
                </table>
            <?php else: ?>
                <div class="text-center py-8 text-gray-500">
                    <i class="fas fa-bell text-4xl mb-2"></i>
                    <p class="text-lg">Tidak ada data reminder yang ditemukan</p>
                    <p class="text-sm">Belum ada reminder yang dibuat untuk kendaraan ini.</p>
                </div>
            <?php endif; ?>
        </div>
    </div>
<?php else: ?>
    <div class="glass-card rounded-xl p-6 fade-in">
        <div class="text-center py-8 text-gray-500">
            <i class="fas fa-search text-4xl mb-2"></i>
            <p class="text-lg">Silakan lakukan pencarian untuk menampilkan data penjualan.</p>
            <p class="text-sm">Masukkan nama pemilik, nomor polisi, nomor rangka, atau nomor mesin.</p>
        </div>
    </div>
<?php endif; ?>

<!-- Modal untuk Reminder -->
<div id="reminderModal" class="fixed inset-0 bg-black bg-opacity-50 hidden z-50 flex items-center justify-center p-4">
    <div class="bg-white rounded-xl shadow-2xl w-full max-w-5xl max-h-[90vh] flex flex-col">
        <div class="flex justify-between items-center p-6 border-b bg-gradient-to-r from-green-50 to-emerald-50 rounded-t-xl">
            <h3 class="text-xl font-bold text-gray-800">
                <i class="fas fa-bell mr-2 text-green-600"></i>
                Buat Reminder
            </h3>
            <button onclick="closeReminderModal()" class="text-gray-500 hover:text-gray-700 bg-white rounded-full p-2 shadow-md transition-all hover:shadow-lg">
                <i class="fas fa-times text-xl"></i>
            </button>
        </div>
        
        <div class="flex-1 overflow-y-auto p-6">
            <div class="bg-gray-50 p-4 rounded-lg mb-6">
                <h4 class="font-semibold text-gray-700 mb-3 border-b pb-2">Data Kendaraan</h4>
                <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4 text-sm">
                    <div>
                        <p class="text-xs text-gray-500">No. Polisi</p>
                        <p class="font-semibold text-gray-800" id="modalPoliceNumber">-</p>
                    </div>
                    <div>
                        <p class="text-xs text-gray-500">Customer Name</p>
                        <p class="font-semibold text-gray-800" id="modalCustomerName">-</p>
                    </div>
                    <div>
                        <p class="text-xs text-gray-500">No. Telepon</p>
                        <p class="font-semibold text-gray-800" id="modalPhoneNumber">-</p>
                    </div>
                    <div class="md:col-span-2 lg:col-span-1">
                        <p class="text-xs text-gray-500">Alamat Pemilik STNK</p>
                        <p class="font-semibold text-gray-800" id="modalAlamatPemilik">-</p>
                    </div>
                    <div>
                        <p class="text-xs text-gray-500">Model Kendaraan</p>
                        <p class="font-semibold text-gray-800" id="modalBasicModel">-</p>
                    </div>
                    <div>
                        <p class="text-xs text-gray-500">Type</p>
                        <p class="font-semibold text-gray-800" id="modalType">-</p>
                    </div>
                    <div>
                        <p class="text-xs text-gray-500">Transmisi</p>
                        <p class="font-semibold text-gray-800" id="modalTransmisi">-</p>
                    </div>
                    <div>
                        <p class="text-xs text-gray-500">Kategori</p>
                        <p class="font-semibold text-gray-800" id="modalKategori">-</p>
                    </div>
                    <div>
                        <p class="text-xs text-gray-500">Tahun</p>
                        <p class="font-semibold text-gray-800" id="modalTahun">-</p>
                    </div>
                    <div>
                        <p class="text-xs text-gray-500">Warna</p>
                        <p class="font-semibold text-gray-800" id="modalWarna">-</p>
                    </div>
                    <div>
                        <p class="text-xs text-gray-500">No. Rangka (VIN)</p>
                        <p class="font-semibold text-gray-800" id="modalVinNumber">-</p>
                    </div>
                    <div>
                        <p class="text-xs text-gray-500">No. Mesin</p>
                        <p class="font-semibold text-gray-800" id="modalNoMesin">-</p>
                    </div>
                    <div>
                        <p class="text-xs text-gray-500">No. Buku Service</p>
                        <p class="font-semibold text-gray-800" id="modalNoBukuService">-</p>
                    </div>
                    <div>
                        <p class="text-xs text-gray-500">Tahun Registrasi</p>
                        <p class="font-semibold text-gray-800" id="modalTahunRegistrasi">-</p>
                    </div>
                </div>
            </div>

            <div class="mb-6">
                <label class="block text-sm font-medium text-gray-700 mb-3">Bisa dihubungi?</label>
                <div class="flex space-x-4">
                    <label class="flex items-center cursor-pointer">
                        <input type="radio" name="contactable" value="yes" class="mr-2 text-green-600 focus:ring-green-500" onchange="handleContactChange(this.value)">
                        <span class="text-gray-700">YA</span>
                    </label>
                    <label class="flex items-center cursor-pointer">
                        <input type="radio" name="contactable" value="no" class="mr-2 text-red-600 focus:ring-red-500" onchange="handleContactChange(this.value)">
                        <span class="text-gray-700">TIDAK</span>
                    </label>
                </div>
            </div>

            <div id="reasonContainer" class="mb-6 hidden">
                <label class="block text-sm font-medium text-gray-700 mb-2">Keterangan</label>
                <select id="reasonSelect" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-green-500" onchange="handleReasonChange(this.value)">
                    <option value="">-- Pilih Keterangan --</option>
                </select>
            </div>

            <div id="bookingDetails" class="hidden space-y-4">
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-2">Tanggal Booking</label>
                    <input type="date" id="bookingDate" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-green-500">
                </div>
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-2">Jam Booking</label>
                    <select id="bookingTime" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-green-500">
                        <option value="">-- Pilih Jam --</option>
                    </select>
                    <p class="text-xs text-gray-500 mt-1">Jam tersedia: 08:00 - 11:00 dan 13:00 - 14:00</p>
                </div>
            </div>
        </div>
        
        <div class="border-t p-4 bg-gray-50 rounded-b-xl">
            <div class="flex justify-end space-x-3">
                <button onclick="closeReminderModal()" class="px-6 py-2 bg-gray-200 hover:bg-gray-300 text-gray-800 font-medium rounded-lg shadow-md transition-all">
                    Batal
                </button>
                <button onclick="saveReminder()" class="px-6 py-2 bg-green-600 hover:bg-green-700 text-white font-medium rounded-lg shadow-md transition-all hover:shadow-lg">
                    <i class="fas fa-save mr-2"></i> Simpan Reminder
                </button>
            </div>
        </div>
    </div>
</div>

<!-- Modal untuk Detail Penjualan per SPK -->
<div id="detailModal" class="fixed inset-0 bg-black bg-opacity-50 hidden z-50 flex items-center justify-center p-4">
    <div class="bg-white rounded-xl shadow-2xl w-full max-w-4xl max-h-[90vh] flex flex-col">
        <div class="flex justify-between items-center p-6 border-b bg-gradient-to-r from-blue-50 to-indigo-50 rounded-t-xl">
            <h3 class="text-xl font-bold text-gray-800">Detail Data Penjualan per SPK</h3>
            <button onclick="closeDetailModal()" class="text-gray-500 hover:text-gray-700 bg-white rounded-full p-2 shadow-md transition-all hover:shadow-lg">
                <i class="fas fa-times text-xl"></i>
            </button>
        </div>
        
        <div id="detailContent" class="flex-1 overflow-y-auto p-6">
        </div>
        
        <div class="border-t p-4 bg-gray-50 rounded-b-xl">
            <div class="flex justify-end">
                <button onclick="closeDetailModal()" class="px-6 py-2 bg-blue-600 hover:bg-blue-700 text-white font-medium rounded-lg shadow-md transition-all hover:shadow-lg">
                    Tutup
                </button>
            </div>
        </div>
    </div>
</div>

<!-- JavaScript -->
<script>
const allPenjualanDetails = <?php echo json_encode($all_details_for_modal); ?>;
const reasonOptions = <?php echo json_encode($reasons_from_db); ?>;
let currentSpkNumber = '';

function showToast(message, type = 'success') {
    const toast = document.createElement('div');
    toast.className = `fixed top-4 right-4 z-50 p-4 rounded-lg shadow-lg text-white text-sm font-medium transform transition-all duration-300 ease-in-out translate-x-full opacity-0`;
    
    if (type === 'success') {
        toast.classList.add('bg-green-500');
    } else if (type === 'error') {
        toast.classList.add('bg-red-500');
    } else if (type === 'info') {
        toast.classList.add('bg-blue-500');
    } else {
        toast.classList.add('bg-gray-800');
    }

    const icon = document.createElement('i');
    icon.className = 'fas fa-check-circle mr-2';
    if (type === 'error') {
        icon.className = 'fas fa-exclamation-circle mr-2';
    } else if (type === 'info') {
        icon.className = 'fas fa-info-circle mr-2';
    }
    toast.appendChild(icon);

    const messageSpan = document.createElement('span');
    messageSpan.textContent = message;
    toast.appendChild(messageSpan);

    document.body.appendChild(toast);

    setTimeout(() => {
        toast.classList.remove('translate-x-full', 'opacity-0');
    }, 100);

    setTimeout(() => {
        toast.classList.add('translate-x-full', 'opacity-0');
        setTimeout(() => {
            document.body.removeChild(toast);
        }, 300);
    }, 4000);
}

document.addEventListener('DOMContentLoaded', function() {
    const searchForm = document.getElementById('searchForm');
    if (searchForm) {
        searchForm.addEventListener('submit', function(e) {
            e.preventDefault();
            
            const searchInput = document.getElementById('searchInput');
            const searchValue = searchInput.value.trim();
            
            if (searchValue !== '') {
                const encodedValue = btoa(unescape(encodeURIComponent(searchValue)));
                window.location.href = '/reminder/search/' + encodedValue;
            } else {
                window.location.href = '/reminder';
            }
        });
    }
});

function openReminderModal(spkNumber, customerName, vinNumber, policeNumber, phoneNumber, alamatPemilik, basicModel, type, transmisi, kategori, tahun, warna, noMesin, noBukuService, tahunRegistrasi) {
    const modal = document.getElementById('reminderModal');
    if (!modal) {
        showToast('Error: Modal reminder tidak ditemukan di halaman!', 'error');
        console.error('Modal element with ID "reminderModal" not found!');
        return;
    }

    try {
        currentSpkNumber = spkNumber;
        
        const radioYes = document.querySelector('input[name="contactable"][value="yes"]');
        const radioNo = document.querySelector('input[name="contactable"][value="no"]');
        if(radioYes) radioYes.checked = false;
        if(radioNo) radioNo.checked = false;

        const reasonSelect = document.getElementById('reasonSelect');
        if(reasonSelect) reasonSelect.innerHTML = '<option value="">-- Pilih Keterangan --</option>';
        
        const reasonContainer = document.getElementById('reasonContainer');
        if(reasonContainer) reasonContainer.classList.add('hidden');
        
        const bookingDetails = document.getElementById('bookingDetails');
        if(bookingDetails) bookingDetails.classList.add('hidden');
        
        document.getElementById('modalPoliceNumber').textContent = policeNumber;
        document.getElementById('modalCustomerName').textContent = customerName;
        document.getElementById('modalPhoneNumber').textContent = phoneNumber;
        document.getElementById('modalAlamatPemilik').textContent = alamatPemilik;
        document.getElementById('modalBasicModel').textContent = basicModel;
        document.getElementById('modalType').textContent = type;
        document.getElementById('modalTransmisi').textContent = transmisi;
        document.getElementById('modalKategori').textContent = kategori;
        document.getElementById('modalTahun').textContent = tahun;
        document.getElementById('modalWarna').textContent = warna;
        document.getElementById('modalVinNumber').textContent = vinNumber;
        document.getElementById('modalNoMesin').textContent = noMesin;
        document.getElementById('modalNoBukuService').textContent = noBukuService;
        document.getElementById('modalTahunRegistrasi').textContent = tahunRegistrasi;
        
        modal.classList.remove('hidden');

    } catch (error) {
        console.error('Terjadi error saat membuka modal reminder:', error);
        showToast('Terjadi kesalahan saat membuka modal. Periksa console browser (F12) untuk detail error.', 'error');
    }
}

function closeReminderModal() {
    document.getElementById('reminderModal').classList.add('hidden');
}

function handleContactChange(value) {
    const reasonContainer = document.getElementById('reasonContainer');
    const reasonSelect = document.getElementById('reasonSelect');
    const bookingDetails = document.getElementById('bookingDetails');
    
    bookingDetails.classList.add('hidden');
    
    if (value === 'yes' || value === 'no') {
        reasonContainer.classList.remove('hidden');
        
        reasonSelect.innerHTML = '<option value="">-- Pilih Keterangan --</option>';
        reasonOptions[value].forEach(reason => {
            const option = document.createElement('option');
            option.value = reason;
            option.textContent = reason;
            reasonSelect.appendChild(option);
        });
    } else {
        reasonContainer.classList.add('hidden');
    }
}

function handleReasonChange(value) {
    const bookingDetails = document.getElementById('bookingDetails');
    
    if (value === 'Booking') {
        bookingDetails.classList.remove('hidden');
        
        const tomorrow = new Date();
        tomorrow.setDate(tomorrow.getDate() + 1);
        const minDate = tomorrow.toISOString().split('T')[0];
        document.getElementById('bookingDate').setAttribute('min', minDate);
        
        document.getElementById('bookingDate').value = minDate;

        populateTimeSlots();
    } else {
        bookingDetails.classList.add('hidden');
    }
}

function populateTimeSlots() {
    const bookingTimeSelect = document.getElementById('bookingTime');
    
    bookingTimeSelect.innerHTML = '<option value="">-- Pilih Jam --</option>';

    const timeSlots = [
        '08:00', '08:30', '09:00', '09:30', '10:00', '10:30', '11:00',
        '13:00', '13:30', '14:00'
    ];

    timeSlots.forEach(time => {
        const option = document.createElement('option');
        option.value = time;
        option.textContent = time;
        bookingTimeSelect.appendChild(option);
    });
}

function saveReminder() {
    const contactable = document.querySelector('input[name="contactable"]:checked');
    const reason = document.getElementById('reasonSelect').value;
    const bookingDate = document.getElementById('bookingDate').value;
    const bookingTime = document.getElementById('bookingTime').value;
    
    if (!contactable) {
        showToast('Silakan pilih YA atau TIDAK.', 'error');
        return;
    }
    
    if (!reason) {
        showToast('Silakan pilih keterangan.', 'error');
        return;
    }
    
    if (reason === 'Booking' && (!bookingDate || !bookingTime)) {
        showToast('Untuk booking, silakan isi tanggal dan jam.', 'error');
        return;
    }

    if (reason === 'Booking') {
        const selectedDate = new Date(bookingDate);
        const today = new Date();
        today.setHours(0, 0, 0, 0);

        if (selectedDate <= today) {
            showToast('Tanggal booking tidak boleh hari ini atau tanggal yang sudah lewat. Silakan pilih tanggal besok atau setelahnya.', 'error');
            return;
        }
    }
    
    const getElementText = (id) => {
        const element = document.getElementById(id);
        return element ? element.textContent : '';
    };
    
    const vehicleData = {
        spkNumber: currentSpkNumber,
        policeNumber: getElementText('modalPoliceNumber'),
        customerName: getElementText('modalCustomerName'),
        phoneNumber: getElementText('modalPhoneNumber'),
        alamatPemilik: getElementText('modalAlamatPemilik'),
        basicModel: getElementText('modalBasicModel'),
        type: getElementText('modalType'),
        transmisi: getElementText('modalTransmisi'),
        kategori: getElementText('modalKategori'),
        tahun: getElementText('modalTahun'),
        warna: getElementText('modalWarna'),
        vinNumber: getElementText('modalVinNumber'),
        noMesin: getElementText('modalNoMesin'),
        noBukuService: getElementText('modalNoBukuService'),
        tahunRegistrasi: getElementText('modalTahunRegistrasi')
    };
    
    const reminderData = {
        ...vehicleData,
        contactable: contactable.value,
        reason: reason,
        bookingDate: bookingDate,
        bookingTime: bookingTime
    };
    
    showToast('Menyimpan reminder...', 'info');
    
    fetch('/save_reminder', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify(reminderData)
    })
    .then(response => {
        const contentType = response.headers.get('content-type');
        if (!contentType || !contentType.includes('application/json')) {
            return response.text().then(text => {
                console.error('Response bukan JSON:', text);
                throw new Error('Server mengembalikan respons bukan JSON. Lihat console untuk detailnya.');
            });
        }
        return response.json();
    })
    .then(data => {
        if (data.success) {
            showToast('Reminder berhasil disimpan!', 'success');
            closeReminderModal();
            
            setTimeout(() => {
                window.location.href = '/reminder';
            }, 2000);

        } else {
            showToast('Gagal menyimpan reminder: ' + data.message, 'error');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showToast('Terjadi kesalahan saat menyimpan reminder: ' + error.message, 'error');
    });
}

function viewPenjualanBySpk(spk) {
    document.getElementById('detailContent').innerHTML = '<div class="flex justify-center items-center py-12"><div class="text-center"><i class="fas fa-spinner fa-spin text-4xl text-blue-600 mb-4"></i><p class="text-gray-600">Memuat data...</p></div></div>';
    document.getElementById('detailModal').classList.remove('hidden');
    
    const data = allPenjualanDetails[spk];

    setTimeout(() => {
        if (data && data.length > 0) {
            const firstItem = data[0];
            
            const categories = {
                jasaDanOpl: { items: [], total: 0, title: 'Jasa dan OPL', icon: 'fa-wrench' },
                oliDanMaterial: { items: [], total: 0, title: 'Oli dan Material', icon: 'fa-oil-can' },
                sparePart: { items: [], total: 0, title: 'Spare Part', icon: 'fa-cogs' },
                lainnya: { items: [], total: 0, title: 'Lainnya', icon: 'fa-box' }
            };
            let grandTotal = 0;

            data.forEach(item => {
                const jumlah = parseFloat(item['QTY']) * parseFloat(item['Harga Satuan'] || 0);
                grandTotal += jumlah;

                const noPart = (item['No. Part'] || '').toUpperCase();
                const namaJasa = item['Nama Jasa'] || '';

                if (namaJasa && !noPart) {
                    categories.jasaDanOpl.items.push(item);
                    categories.jasaDanOpl.total += jumlah;
                } else if (noPart.startsWith('SGO') || noPart.startsWith('MTS') || noPart.startsWith('MTL')) {
                    categories.oliDanMaterial.items.push(item);
                    categories.oliDanMaterial.total += jumlah;
                } else if (noPart.startsWith('SGP') || noPart.startsWith('SPL')) {
                    categories.sparePart.items.push(item);
                    categories.sparePart.total += jumlah;
                } else {
                    categories.lainnya.items.push(item);
                    categories.lainnya.total += jumlah;
                }
            });

            let html = `
                <div class="grid grid-cols-1 md:grid-cols-3 gap-4 mb-6">
                    <div class="bg-gradient-to-r from-blue-50 to-blue-100 p-4 rounded-lg shadow-sm border border-blue-200">
                        <div class="flex items-center">
                            <div class="p-2 bg-blue-500 rounded-lg mr-3">
                                <i class="fas fa-file-alt text-white"></i>
                            </div>
                            <div>
                                <p class="text-xs text-gray-600">No. SPK</p>
                                <p class="font-bold text-gray-800">${firstItem['No. SPK']}</p>
                            </div>
                        </div>
                    </div>
                    <div class="bg-gradient-to-r from-green-50 to-green-100 p-4 rounded-lg shadow-sm border border-green-200">
                        <div class="flex items-center">
                            <div class="p-2 bg-green-500 rounded-lg mr-3">
                                <i class="fas fa-user text-white"></i>
                            </div>
                            <div>
                                <p class="text-xs text-gray-600">Nama Customer</p>
                                <p class="font-bold text-gray-800">${firstItem['Nama Customer'] || '-'}</p>
                            </div>
                        </div>
                    </div>
                    <div class="bg-gradient-to-r from-purple-50 to-purple-100 p-4 rounded-lg shadow-sm border border-purple-200">
                        <div class="flex items-center">
                            <div class="p-2 bg-purple-500 rounded-lg mr-3">
                                <i class="fas fa-car text-white"></i>
                            </div>
                            <div>
                                <p class="text-xs text-gray-600">No. Polisi</p>
                                <p class="font-bold text-gray-800">${firstItem['No. Polisi']}</p>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="space-y-6">
            `;
            
            for (const key in categories) {
                const category = categories[key];
                if (category.items.length > 0) {
                    html += `
                        <div class="bg-white rounded-lg shadow-md border border-gray-200 overflow-hidden">
                            <div class="bg-gradient-to-r from-gray-50 to-gray-100 px-4 py-3 border-b">
                                <div class="flex items-center">
                                    <i class="fas ${category.icon} mr-2 text-gray-700"></i>
                                    <h4 class="font-semibold text-gray-800">${category.title}</h4>
                                    <span class="ml-auto bg-gray-200 text-gray-700 text-xs px-2 py-1 rounded-full">${category.items.length} item</span>
                                </div>
                            </div>
                            <div class="overflow-x-auto">
                                <table class="w-full text-sm">
                                    <thead class="bg-gray-50">
                                        <tr>
                                            <th class="p-3 text-left font-medium text-gray-700">Nama Part / Jasa</th>
                                            <th class="p-3 text-left font-medium text-gray-700">No. Part</th>
                                            <th class="p-3 text-center font-medium text-gray-700">QTY</th>
                                            <th class="p-3 text-right font-medium text-gray-700">Harga Satuan</th>
                                            <th class="p-3 text-right font-medium text-gray-700">Jumlah</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                    `;
                    category.items.forEach((item, index) => {
                        const jumlah = parseFloat(item['QTY']) * parseFloat(item['Harga Satuan'] || 0);
                        html += `
                            <tr class="${index % 2 === 0 ? 'bg-white' : 'bg-gray-50'} hover:bg-blue-50 transition-colors">
                                <td class="p-3 font-medium">${item['Nama Part'] || item['Nama Jasa'] || '-'}</td>
                                <td class="p-3">${item['No. Part'] || '-'}</td>
                                <td class="p-3 text-center">${item['QTY'] || '-'}</td>
                                <td class="p-3 text-right">Rp ${parseFloat(item['Harga Satuan'] || 0).toLocaleString('id-ID')}</td>
                                <td class="p-3 text-right font-semibold">Rp ${jumlah.toLocaleString('id-ID')}</td>
                            </tr>
                        `;
                    });
                    html += `
                                    </tbody>
                                </table>
                            </div>
                            <div class="bg-gray-50 px-4 py-3 border-t flex justify-end">
                                <div class="font-semibold text-gray-800">
                                    Subtotal ${category.title}: <span class="text-blue-600">Rp ${category.total.toLocaleString('id-ID')}</span>
                                </div>
                            </div>
                        </div>
                    `;
                }
            }

            html += `
                </div>
                
                <div class="mt-6 p-6 bg-gradient-to-r from-blue-50 to-indigo-50 rounded-xl shadow-md border border-blue-200">
                    <div class="flex justify-between items-center">
                        <h4 class="font-bold text-gray-800 text-lg">Total Keseluruhan</h4>
                        <div class="text-2xl font-bold text-blue-600">
                            Rp ${grandTotal.toLocaleString('id-ID')}
                        </div>
                    </div>
                </div>
            `;
            
            document.getElementById('detailContent').innerHTML = html;
        } else {
            document.getElementById('detailContent').innerHTML = '<div class="flex flex-col items-center justify-center py-12 text-red-500"><i class="fas fa-exclamation-circle text-4xl mb-4"></i><p class="text-lg font-medium">Gagal memuat data atau data tidak ditemukan.</p></div>';
        }
    }, 500);
}

function closeDetailModal() {
    document.getElementById('detailModal').classList.add('hidden');
}
</script>

<?php require_once '../includes/footer.php'; ?>