<?php
 $page_title = "Jadwal Service";
 $page_subtitle = "Kelola jadwal service kendaraan";

require_once '../includes/header.php';

// Get filter parameters
 $date = isset($_GET['date']) ? sanitize_input($_GET['date']) : date('Y-m-d');
 $technician = isset($_GET['technician']) ? sanitize_input($_GET['technician']) : '';

// Build query
 $sql = "SELECT s.*, c.name as customer_name, v.model, v.license_plate, t.name as technician_name 
        FROM services s 
        JOIN customers c ON s.customer_id = c.id 
        JOIN vehicles v ON s.vehicle_id = v.id 
        LEFT JOIN technicians t ON s.technician_id = t.id 
        WHERE s.service_date = ? AND s.status != 'cancelled'";

 $params = [$date];
 $types = "s";

if (!empty($technician)) {
    $sql .= " AND s.technician_id = ?";
    $params[] = $technician;
    $types .= "i";
}

 $sql .= " ORDER BY s.service_time ASC";

// Prepare and execute query
 $stmt = $conn->prepare($sql);
 $stmt->bind_param($types, ...$params);
 $stmt->execute();
 $result = $stmt->get_result();

// Get technicians for filter
 $techniciansResult = $conn->query("SELECT id, name FROM technicians ORDER BY name");
 $technicians = [];
if ($techniciansResult->num_rows > 0) {
    while ($row = $techniciansResult->fetch_assoc()) {
        $technicians[] = $row;
    }
}

// Get service statistics for the selected date
 $totalServices = $result->num_rows;
 $completedServices = 0;
 $inProgressServices = 0;

if ($totalServices > 0) {
    // Reset result pointer and count by status
    $result->data_seek(0);
    while ($row = $result->fetch_assoc()) {
        if ($row['status'] == 'completed') {
            $completedServices++;
        } else if ($row['status'] == 'in-progress') {
            $inProgressServices++;
        }
    }
    // Reset result pointer again for display
    $result->data_seek(0);
}
?>
<!-- Filter Section -->
<div class="glass-card rounded-xl p-6 mb-6 fade-in">
    <div class="flex flex-col md:flex-row md:items-center md:justify-between mb-4">
        <h2 class="text-lg font-semibold text-gray-800">Filter Jadwal</h2>
        <button class="btn-primary px-3 py-1 text-white rounded-lg text-sm mt-2 md:mt-0" onclick="addNewService()">
            <i class="fas fa-plus mr-1"></i> Tambah Jadwal
        </button>
    </div>
    
    <form method="GET" action="/pages/jadwal-service.php" class="grid grid-cols-1 md:grid-cols-3 gap-4">
        <div>
            <label class="block text-sm font-medium text-gray-700 mb-1">Tanggal</label>
            <input type="date" name="date" value="<?php echo $date; ?>" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-suzuki-blue">
        </div>
        
        <div>
            <label class="block text-sm font-medium text-gray-700 mb-1">Teknisi</label>
            <select name="technician" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-suzuki-blue">
                <option value="">Semua Teknisi</option>
                <?php foreach ($technicians as $t): ?>
                <option value="<?php echo $t['id']; ?>" <?php echo ($technician == $t['id']) ? 'selected' : ''; ?>><?php echo $t['name']; ?></option>
                <?php endforeach; ?>
            </select>
        </div>
        
        <div class="flex items-end">
            <button type="submit" class="w-full btn-primary px-4 py-2 text-white rounded-lg">
                <i class="fas fa-search mr-2"></i> Cari
            </button>
        </div>
    </form>
</div>

<!-- Statistics Cards -->
<div class="grid grid-cols-1 md:grid-cols-3 gap-6 mb-6">
    <div class="glass-card rounded-xl p-5 fade-in">
        <div class="flex items-center justify-between mb-4">
            <div class="stat-icon bg-blue-100">
                <i class="fas fa-calendar-check text-suzuki-blue text-xl"></i>
            </div>
        </div>
        <h3 class="text-2xl font-bold text-gray-800"><?php echo $totalServices; ?></h3>
        <p class="text-gray-600 text-sm">Total Service</p>
    </div>
    
    <div class="glass-card rounded-xl p-5 fade-in" style="animation-delay: 0.1s">
        <div class="flex items-center justify-between mb-4">
            <div class="stat-icon bg-yellow-100">
                <i class="fas fa-wrench text-yellow-600 text-xl"></i>
            </div>
        </div>
        <h3 class="text-2xl font-bold text-gray-800"><?php echo $inProgressServices; ?></h3>
        <p class="text-gray-600 text-sm">Dalam Proses</p>
    </div>
    
    <div class="glass-card rounded-xl p-5 fade-in" style="animation-delay: 0.2s">
        <div class="flex items-center justify-between mb-4">
            <div class="stat-icon bg-green-100">
                <i class="fas fa-check-circle text-green-600 text-xl"></i>
            </div>
        </div>
        <h3 class="text-2xl font-bold text-gray-800"><?php echo $completedServices; ?></h3>
        <p class="text-gray-600 text-sm">Selesai</p>
    </div>
</div>

<!-- Schedule Table -->
<div class="glass-card rounded-xl p-6 fade-in">
    <div class="flex items-center justify-between mb-4">
        <h2 class="text-lg font-semibold text-gray-800">Jadwal Service - <?php echo date('d F Y', strtotime($date)); ?></h2>
        <div class="flex space-x-2">
            <button class="px-3 py-1 text-sm bg-gray-100 text-gray-600 rounded-lg" onclick="changeDate('prev')">
                <i class="fas fa-chevron-left"></i>
            </button>
            <button class="px-3 py-1 text-sm bg-gray-100 text-gray-600 rounded-lg" onclick="changeDate('today')">Hari Ini</button>
            <button class="px-3 py-1 text-sm bg-gray-100 text-gray-600 rounded-lg" onclick="changeDate('next')">
                <i class="fas fa-chevron-right"></i>
            </button>
        </div>
    </div>
    
    <div class="overflow-x-auto">
        <table class="w-full">
            <thead>
                <tr class="text-left text-xs text-gray-500 uppercase tracking-wider border-b">
                    <th class="pb-3">Waktu</th>
                    <th class="pb-3">Konsumen</th>
                    <th class="pb-3">Kendaraan</th>
                    <th class="pb-3">Layanan</th>
                    <th class="pb-3">Teknisi</th>
                    <th class="pb-3">Status</th>
                    <th class="pb-3">Aksi</th>
                </tr>
            </thead>
            <tbody class="text-sm">
                <?php if ($result->num_rows > 0): ?>
                    <?php while ($row = $result->fetch_assoc()): ?>
                    <tr class="table-row border-b">
                        <td class="py-3">
                            <p class="text-gray-800"><?php echo date('H:i', strtotime($row['service_time'])); ?></p>
                        </td>
                        <td class="py-3">
                            <div class="flex items-center">
                                <img src="https://picsum.photos/seed/<?php echo $row['customer_name']; ?>/32/32.jpg" alt="User" class="w-8 h-8 rounded-lg mr-2">
                                <div>
                                    <p class="font-medium text-gray-800"><?php echo $row['customer_name']; ?></p>
                                </div>
                            </div>
                        </td>
                        <td class="py-3">
                            <p class="font-medium text-gray-800"><?php echo $row['model']; ?></p>
                            <p class="text-xs text-gray-500"><?php echo $row['license_plate']; ?></p>
                        </td>
                        <td class="py-3">
                            <p class="text-gray-800"><?php echo $row['type']; ?></p>
                        </td>
                        <td class="py-3">
                            <p class="text-gray-800"><?php echo $row['technician_name'] ? $row['technician_name'] : 'Belum ditugaskan'; ?></p>
                        </td>
                        <td class="py-3">
                            <?php
                            $statusClass = '';
                            $statusText = '';
                            
                            switch ($row['status']) {
                                case 'scheduled':
                                    $statusClass = 'bg-blue-100 text-blue-700';
                                    $statusText = 'Terjadwal';
                                    break;
                                case 'in-progress':
                                    $statusClass = 'bg-yellow-100 text-yellow-700';
                                    $statusText = 'Dalam Proses';
                                    break;
                                case 'completed':
                                    $statusClass = 'bg-green-100 text-green-700';
                                    $statusText = 'Selesai';
                                    break;
                            }
                            ?>
                            <span class="badge px-2 py-1 text-xs rounded-full <?php echo $statusClass; ?>"><?php echo $statusText; ?></span>
                        </td>
                        <td class="py-3">
                            <div class="flex space-x-2">
                                <button class="text-suzuki-blue hover:text-suzuki-dark" onclick="viewService(<?php echo $row['id']; ?>)">
                                    <i class="fas fa-eye"></i>
                                </button>
                                <button class="text-green-600 hover:text-green-800" onclick="editService(<?php echo $row['id']; ?>)">
                                    <i class="fas fa-edit"></i>
                                </button>
                                <button class="text-red-600 hover:text-red-800" onclick="deleteService(<?php echo $row['id']; ?>)">
                                    <i class="fas fa-trash"></i>
                                </button>
                            </div>
                        </td>
                    </tr>
                    <?php endwhile; ?>
                <?php else: ?>
                    <tr>
                        <td colspan="7" class="py-8 text-center text-gray-500">
                            <i class="fas fa-calendar-times text-4xl mb-2"></i>
                            <p>Tidak ada jadwal service pada tanggal ini</p>
                        </td>
                    </tr>
                <?php endif; ?>
            </tbody>
        </table>
    </div>
</div>

<script>
function changeDate(direction) {
    const currentDate = new Date('<?php echo $date; ?>');
    let newDate;
    
    if (direction === 'prev') {
        newDate = new Date(currentDate);
        newDate.setDate(currentDate.getDate() - 1);
    } else if (direction === 'next') {
        newDate = new Date(currentDate);
        newDate.setDate(currentDate.getDate() + 1);
    } else if (direction === 'today') {
        newDate = new Date();
    }
    
    const formattedDate = newDate.toISOString().split('T')[0];
    window.location.href = `/pages/jadwal-service.php?date=${formattedDate}`;
}
</script>

<?php require_once '../includes/footer.php'; ?>