<?php
 $page_title = "Manajemen Alasan Reminder";
 $page_subtitle = "Tambah, edit, atau hapus alasan untuk dropdown reminder";

require_once '../includes/header.php';

// --- PROSES PENANGANAN FORM (CRUD) ---
// (Bagian ini sama, tidak ada perubahan)
if (isset($_POST['action']) && $_POST['action'] == 'add') {
    $reason_text = sanitize_input($_POST['reason_text']);
    $category = sanitize_input($_POST['category']);
    if (!empty($reason_text) && in_array($category, ['yes', 'no'])) {
        $sql = "INSERT INTO reminder_reasons (reason_text, category) VALUES (?, ?)";
        $stmt = $conn->prepare($sql);
        $stmt->bind_param("ss", $reason_text, $category);
        if ($stmt->execute()) {
            $success_message = "Alasan baru berhasil ditambahkan.";
        } else {
            $error_message = "Gagal menambahkan alasan: " . $stmt->error;
        }
        $stmt->close();
    } else {
        $error_message = "Data tidak lengkap atau kategori tidak valid.";
    }
}

if (isset($_POST['action']) && $_POST['action'] == 'edit') {
    $id = (int)$_POST['id'];
    $reason_text = sanitize_input($_POST['reason_text']);
    $category = sanitize_input($_POST['category']);
    if ($id > 0 && !empty($reason_text) && in_array($category, ['yes', 'no'])) {
        $sql = "UPDATE reminder_reasons SET reason_text = ?, category = ? WHERE id = ?";
        $stmt = $conn->prepare($sql);
        $stmt->bind_param("ssi", $reason_text, $category, $id);
        if ($stmt->execute()) {
            $success_message = "Alasan berhasil diperbarui.";
        } else {
            $error_message = "Gagal memperbarui alasan: " . $stmt->error;
        }
        $stmt->close();
    } else {
        $error_message = "Data tidak lengkap atau ID tidak valid.";
    }
}

if (isset($_POST['action']) && $_POST['action'] == 'delete') {
    $id = (int)$_POST['delete_id'];
    if ($id > 0) {
        $sql = "DELETE FROM reminder_reasons WHERE id = ?";
        $stmt = $conn->prepare($sql);
        $stmt->bind_param("i", $id);
        if ($stmt->execute()) {
            $success_message = "Alasan berhasil dihapus.";
        } else {
            $error_message = "Gagal menghapus alasan: " . $stmt->error;
        }
        $stmt->close();
    }
}

if (isset($_POST['action']) && $_POST['action'] == 'toggle_status') {
    $id = (int)$_POST['id'];
    $is_active = (int)$_POST['is_active'];
    if ($id > 0) {
        $sql = "UPDATE reminder_reasons SET is_active = ? WHERE id = ?";
        $stmt = $conn->prepare($sql);
        $stmt->bind_param("ii", $is_active, $id);
        if ($stmt->execute()) {
            echo json_encode(['success' => true, 'message' => 'Status berhasil diperbarui.']);
        } else {
            echo json_encode(['success' => false, 'message' => 'Gagal memperbarui status.']);
        }
        $stmt->close();
        exit;
    }
}

// --- AMBIL SEMUA DATA DARI DATABASE ---
 $sql = "SELECT id, reason_text, category, is_active FROM reminder_reasons ORDER BY category, reason_text";
 $result = $conn->query($sql);
 $all_reasons = [];
if ($result) {
    $all_reasons = $result->fetch_all(MYSQLI_ASSOC);
}
?>

<!-- Main Content -->
<div class="glass-card rounded-xl p-6 mb-6 fade-in">
    <div class="flex flex-col md:flex-row md:items-center md:justify-between mb-6">
        <div>
            <h2 class="text-xl font-bold text-gray-800">Daftar Alasan Reminder</h2>
            <p class="text-sm text-gray-500">Kelola alasan yang akan muncul di dropdown form reminder.</p>
        </div>
    </div>

    <!-- Tampilkan Pesan Sukses atau Error -->
    <?php if (isset($success_message)): ?>
        <div class="mb-4 p-4 bg-green-100 border border-green-400 text-green-700 rounded-lg">
            <?php echo $success_message; ?>
        </div>
    <?php endif; ?>
    <?php if (isset($error_message)): ?>
        <div class="mb-4 p-4 bg-red-100 border border-red-400 text-red-700 rounded-lg">
            <?php echo $error_message; ?>
        </div>
    <?php endif; ?>

    <!-- Form Tambah Alasan Baru -->
    <div class="bg-gray-50 p-4 rounded-lg mb-6">
        <h3 class="text-lg font-semibold text-gray-700 mb-3">Tambah Alasan Baru</h3>
        <form action="alasan_reminder" method="POST" class="grid grid-cols-1 md:grid-cols-12 gap-3">
            <input type="hidden" name="action" value="add">
            <div class="md:col-span-6">
                <label for="reason_text" class="block text-sm font-medium text-gray-700 mb-1">Teks Alasan</label>
                <input type="text" id="reason_text" name="reason_text" required
                       class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
                       placeholder="Contoh: Tunggu Promo">
            </div>
            <div class="md:col-span-3">
                <label for="category" class="block text-sm font-medium text-gray-700 mb-1">Kategori</label>
                <select id="category" name="category" required
                        class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
                    <option value="yes">Bisa Dihubungi (YA)</option>
                    <option value="no">Tidak Bisa Dihubungi (TIDAK)</option>
                </select>
            </div>
            <div class="md:col-span-3 flex items-end">
                <button type="submit" class="w-full px-4 py-2 bg-blue-600 hover:bg-blue-700 text-white font-semibold rounded-lg shadow-md transition duration-150 ease-in-out">
                    <i class="fas fa-plus mr-2"></i> Tambah Alasan
                </button>
            </div>
        </form>
    </div>

    <!-- Tabel Daftar Alasan -->
    <div class="overflow-x-auto">
        <table id="reasonsTable" class="w-full text-sm display">
            <thead>
                <tr class="text-left text-xs text-gray-500 uppercase tracking-wider border-b bg-gray-50">
                    <th class="p-3">No</th>
                    <th class="p-3">Alasan</th>
                    <th class="p-3">Kategori</th>
                    <th class="p-3 text-center">Status</th>
                    <th class="p-3 text-center">Aksi</th>
                </tr>
            </thead>
            <tbody>
                <?php if (!empty($all_reasons)): ?>
                    <?php foreach ($all_reasons as $index => $row): ?>
                        <tr>
                            <td class="p-3"><?php echo $index + 1; ?></td>
                            <td class="p-3 font-medium"><?php echo htmlspecialchars($row['reason_text']); ?></td>
                            <td class="p-3">
                                <span class="px-2 py-1 text-xs rounded-full <?php echo $row['category'] == 'yes' ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'; ?>">
                                    <?php echo ucfirst($row['category']); ?>
                                </span>
                            </td>
                            <td class="p-3 text-center">
                                <label class="relative inline-flex items-center cursor-pointer">
                                    <input type="checkbox" class="sr-only peer status-toggle" 
                                           data-id="<?php echo $row['id']; ?>"
                                           <?php echo $row['is_active'] ? 'checked' : ''; ?>>
                                    <div class="w-11 h-6 bg-gray-200 peer-focus:outline-none peer-focus:ring-4 peer-focus:ring-blue-300 rounded-full peer peer-checked:after:translate-x-full peer-checked:after:border-white after:content-[''] after:absolute after:top-[2px] after:left-[2px] after:bg-white after:rounded-full after:h-5 after:w-5 after:transition-all peer-checked:bg-blue-600"></div>
                                </label>
                            </td>
                            <td class="p-3 text-center">
                                <button onclick="openEditModal(<?php echo $row['id']; ?>, '<?php echo htmlspecialchars($row['reason_text'], ENT_QUOTES); ?>', '<?php echo $row['category']; ?>')" 
                                        class="text-yellow-600 hover:text-yellow-800 mr-2" title="Edit">
                                    <i class="fas fa-edit"></i>
                                </button>
                                <button onclick="confirmDelete(<?php echo $row['id']; ?>, '<?php echo htmlspecialchars($row['reason_text'], ENT_QUOTES); ?>')" 
                                        class="text-red-600 hover:text-red-800" title="Hapus">
                                    <i class="fas fa-trash"></i>
                                </button>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                <?php else: ?>
                    <tr>
                        <td colspan="5" class="p-8 text-center text-gray-500">
                            <i class="fas fa-inbox text-4xl mb-2"></i>
                            <p>Belum ada data alasan.</p>
                        </td>
                    </tr>
                <?php endif; ?>
            </tbody>
        </table>
    </div>
</div>

<!-- Modal Edit & Hapus (Tidak Berubah) -->
<div id="editModal" class="fixed inset-0 bg-black bg-opacity-50 hidden z-50 flex items-center justify-center p-4">
    <div class="bg-white rounded-xl shadow-2xl w-full max-w-md">
        <div class="flex justify-between items-center p-4 border-b">
            <h3 class="text-lg font-bold text-gray-800">Edit Alasan</h3>
            <button onclick="closeEditModal()" class="text-gray-500 hover:text-gray-700">
                <i class="fas fa-times"></i>
            </button>
        </div>
        <form action="alasan_reminder" method="POST" class="p-4">
            <input type="hidden" name="action" value="edit">
            <input type="hidden" id="editId" name="id">
            <div class="mb-4">
                <label for="editReasonText" class="block text-sm font-medium text-gray-700 mb-1">Teks Alasan</label>
                <input type="text" id="editReasonText" name="reason_text" required
                       class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
            </div>
            <div class="mb-4">
                <label class="block text-sm font-medium text-gray-700 mb-1">Kategori</label>
                <div class="flex space-x-4">
                    <label class="flex items-center">
                        <input type="radio" name="category" value="yes" required class="mr-2">
                        <span>YA</span>
                    </label>
                    <label class="flex items-center">
                        <input type="radio" name="category" value="no" required class="mr-2">
                        <span>TIDAK</span>
                    </label>
                </div>
            </div>
            <div class="flex justify-end space-x-2">
                <button type="button" onclick="closeEditModal()" class="px-4 py-2 bg-gray-200 hover:bg-gray-300 text-gray-800 rounded-lg">
                    Batal
                </button>
                <button type="submit" class="px-4 py-2 bg-blue-600 hover:bg-blue-700 text-white rounded-lg">
                    Simpan Perubahan
                </button>
            </div>
        </form>
    </div>
</div>

<div id="deleteModal" class="fixed inset-0 bg-black bg-opacity-50 hidden z-50 flex items-center justify-center p-4">
    <div class="bg-white rounded-xl shadow-2xl w-full max-w-sm p-6">
        <div class="text-center">
            <i class="fas fa-exclamation-triangle text-5xl text-red-500 mb-4"></i>
            <h3 class="text-lg font-bold text-gray-800 mb-2">Konfirmasi Hapus</h3>
            <p class="text-gray-600 mb-6">Apakah Anda yakin ingin menghapus alasan "<span id="deleteReasonName" class="font-semibold"></span>"?</p>
        </div>
        <form action="alasan_reminder" method="POST">
            <input type="hidden" name="action" value="delete">
            <input type="hidden" id="deleteId" name="delete_id">
            <div class="flex justify-center space-x-2">
                <button type="button" onclick="closeDeleteModal()" class="px-4 py-2 bg-gray-200 hover:bg-gray-300 text-gray-800 rounded-lg">
                    Batal
                </button>
                <button type="submit" class="px-4 py-2 bg-red-600 hover:bg-red-700 text-white rounded-lg">
                    Hapus
                </button>
            </div>
        </form>
    </div>
</div>

<!-- Include DataTables CSS & JS -->
<link rel="stylesheet" href="https://cdn.datatables.net/1.13.6/css/dataTables.tailwindcss.min.css">
<script src="https://code.jquery.com/jquery-3.7.0.js"></script>
<script src="https://cdn.datatables.net/1.13.6/js/jquery.dataTables.min.js"></script>

<script>
 $(document).ready(function() {
    // Inisialisasi DataTables yang sangat sederhana
    $('#reasonsTable').DataTable({
        language: {
            url: "//cdn.datatables.net/plug-ins/1.13.6/i18n/id.json" // Bahasa Indonesia
        },
        responsive: true,
        pageLength: 10, // Default 10 item per halaman
        lengthMenu: [ [10, 25, 50, -1], [10, 25, 50, "Semua"] ] // Opsi jumlah item per halaman
    });

    // Gunakan Event Delegation untuk toggle status
    $('#reasonsTable').on('change', '.status-toggle', function() {
        const id = $(this).data('id');
        const isActive = $(this).is(':checked');

        $.ajax({
            url: 'alasan_reminder',
            method: 'POST',
            data: {
                action: 'toggle_status',
                id: id,
                is_active: isActive ? 1 : 0
            },
            dataType: 'json',
            success: function(response) {
                if (response.success) {
                    showToast(response.message, 'success');
                } else {
                    showToast(response.message, 'error');
                    // Kembalikan toggle ke posisi semula
                    location.reload(); // Sederhana dan aman
                }
            },
            error: function() {
                showToast('Terjadi kesalahan saat memperbarui status.', 'error');
                location.reload();
            }
        });
    });
});

// --- Fungsi Modal & Toast (Tidak Berubah) ---
function openEditModal(id, reason, category) {
    document.getElementById('editId').value = id;
    document.getElementById('editReasonText').value = reason;
    document.querySelector(`input[name="category"][value="${category}"]`).checked = true;
    document.getElementById('editModal').classList.remove('hidden');
}
function closeEditModal() { document.getElementById('editModal').classList.add('hidden'); }
function confirmDelete(id, reason) {
    document.getElementById('deleteId').value = id;
    document.getElementById('deleteReasonName').textContent = reason;
    document.getElementById('deleteModal').classList.remove('hidden');
}
function closeDeleteModal() { document.getElementById('deleteModal').classList.add('hidden'); }

function showToast(message, type = 'success') {
    const toast = document.createElement('div');
    toast.className = `fixed top-4 right-4 z-50 p-4 rounded-lg shadow-lg text-white text-sm font-medium transform transition-all duration-300 ease-in-out translate-x-full opacity-0`;
    if (type === 'success') toast.classList.add('bg-green-500');
    else if (type === 'error') toast.classList.add('bg-red-500');
    else toast.classList.add('bg-gray-800');
    const icon = document.createElement('i');
    icon.className = type === 'error' ? 'fas fa-exclamation-circle mr-2' : 'fas fa-check-circle mr-2';
    toast.appendChild(icon);
    const messageSpan = document.createElement('span');
    messageSpan.textContent = message;
    toast.appendChild(messageSpan);
    document.body.appendChild(toast);
    setTimeout(() => { toast.classList.remove('translate-x-full', 'opacity-0'); }, 100);
    setTimeout(() => {
        toast.classList.add('translate-x-full', 'opacity-0');
        setTimeout(() => { document.body.removeChild(toast); }, 300);
    }, 4000);
}
</script>

<?php require_once '../includes/footer.php'; ?>