<?php
// 启动会话
session_start();

// 设置JSON响应头
header('Content-Type: application/json');

// 引入数据库配置
require_once '../config/database.php';

// 定义清理输入函数
function sanitize_input($data) {
    $data = trim($data);
    $data = stripslashes($data);
    $data = htmlspecialchars($data);
    return $data;
}

// 响应函数
function send_response($success, $message, $data = null) {
    $response = [
        'success' => $success,
        'message' => $message
    ];
    
    if ($data !== null) {
        $response['data'] = $data;
    }
    
    echo json_encode($response);
    exit;
}

// 检查请求方法
if ($_SERVER["REQUEST_METHOD"] != "POST") {
    send_response(false, 'Metode request tidak valid');
}

// 检查是否为AJAX请求
if (!isset($_SERVER['HTTP_X_REQUESTED_WITH']) || strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) != 'xmlhttprequest') {
    send_response(false, 'Akses tidak diizinkan');
}

// 获取并验证输入
 $username = isset($_POST['username']) ? sanitize_input($_POST['username']) : '';
 $password = isset($_POST['password']) ? $_POST['password'] : '';

// 验证输入不为空
if (empty($username) || empty($password)) {
    send_response(false, 'Username dan password harus diisi');
}

try {
    // 查询用户信息 - 包含所有必要字段
    $sql = "SELECT id, username, name, email, password, role, phone, status, last_login FROM users WHERE (username = ? OR email = ?) AND status = 'active'";
    $stmt = $conn->prepare($sql);
    
    if (!$stmt) {
        throw new Exception("Database query preparation failed: " . $conn->error);
    }
    
    $stmt->bind_param("ss", $username, $username);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows === 0) {
        // 记录失败尝试
        log_failed_attempt($username, 'User not found');
        send_response(false, 'Username atau password salah');
    }
    
    $user = $result->fetch_assoc();
    
    // 验证密码
    if (!password_verify($password, $user['password'])) {
        // 记录失败尝试
        log_failed_attempt($username, 'Invalid password');
        send_response(false, 'Username atau password salah');
    }
    
    // 检查账户状态
    if ($user['status'] !== 'active') {
        send_response(false, 'Akun Anda tidak aktif. Silakan hubungi administrator.');
    }
    
    // 更新最后登录时间
    $update_sql = "UPDATE users SET last_login = NOW() WHERE id = ?";
    $update_stmt = $conn->prepare($update_sql);
    $update_stmt->bind_param("i", $user['id']);
    $update_stmt->execute();
    
    // 设置会话变量
    $_SESSION['user_id'] = $user['id'];
    $_SESSION['username'] = $user['username'];
    $_SESSION['name'] = $user['name'];
    $_SESSION['email'] = $user['email'];
    $_SESSION['role'] = $user['role'];
    $_SESSION['phone'] = $user['phone'];
    $_SESSION['login_time'] = time();
    $_SESSION['last_login'] = $user['last_login'];
    
    // 记录成功登录
    log_successful_login($user['id']);
    
    // 根据角色重定向到不同的仪表板
    $redirect_url = '/dashboard';
    switch ($user['role']) {
        case 'Service Manager':
            $redirect_url = '/dashboard';
            break;
        case 'Customer Relation Officer':
            $redirect_url = '/dashboard';
            break;
        case 'Service Relation Officer':
            $redirect_url = '/dashboard';
            break;
        case 'Service Advisor':
            $redirect_url = '/dashboard';
            break;
        case 'admin':
            $redirect_url = '/admin/dashboard';
            break;
    }
    
    // 返回成功响应
    send_response(true, 'Login berhasil', [
        'redirect_url' => $redirect_url,
        'user' => [
            'id' => $user['id'],
            'name' => $user['name'],
            'role' => $user['role'],
            'email' => $user['email']
        ]
    ]);
    
} catch (Exception $e) {
    // 记录错误
    error_log("Login error: " . $e->getMessage());
    send_response(false, 'Terjadi kesalahan sistem. Silakan coba lagi.');
}

// 记录失败登录尝试的函数
function log_failed_attempt($username, $reason) {
    global $conn;
    
    try {
        $ip_address = $_SERVER['REMOTE_ADDR'];
        $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? '';
        
        $sql = "INSERT INTO failed_login_logs (username, reason, ip_address, user_agent, attempt_time) VALUES (?, ?, ?, ?, NOW())";
        $stmt = $conn->prepare($sql);
        $stmt->bind_param("ssss", $username, $reason, $ip_address, $user_agent);
        $stmt->execute();
    } catch (Exception $e) {
        error_log("Failed to log failed attempt: " . $e->getMessage());
    }
}

// 记录成功登录的函数
function log_successful_login($user_id) {
    global $conn;
    
    try {
        $ip_address = $_SERVER['REMOTE_ADDR'];
        $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? '';
        
        $sql = "INSERT INTO login_logs (user_id, ip_address, user_agent, login_time) VALUES (?, ?, ?, NOW())";
        $stmt = $conn->prepare($sql);
        $stmt->bind_param("iss", $user_id, $ip_address, $user_agent);
        $stmt->execute();
    } catch (Exception $e) {
        error_log("Failed to log successful login: " . $e->getMessage());
    }
}
?>